<?php
date_default_timezone_set('Asia/Colombo');
include '../config.php';
header('Content-Type: application/json');

$commission_rate = 0.01; // keep in sync with process_sale.php


$input = json_decode(file_get_contents('php://input'), true);
$item_id = isset($input['item_id']) ? intval($input['item_id']) : 0;
$sale_id = isset($input['sale_id']) ? intval($input['sale_id']) : 0;
$new_product_id = isset($input['new_product_id']) ? intval($input['new_product_id']) : 0;
$new_quantity = isset($input['new_quantity']) ? intval($input['new_quantity']) : 1;
$price_diff = isset($input['price_diff']) ? floatval($input['price_diff']) : 0;

if (!$item_id || !$sale_id || !$new_product_id || !$new_quantity) {
    echo json_encode(['success' => false, 'message' => 'Missing data for exchange']);
    exit;
}

try {
    $conn->begin_transaction();

    // Ensure exchanged column exists
    $colCheck = $conn->query("SHOW COLUMNS FROM sale_items LIKE 'exchanged'");
    if ($colCheck && $colCheck->num_rows === 0) {
        $conn->query("ALTER TABLE sale_items ADD COLUMN exchanged TINYINT(1) NOT NULL DEFAULT 0");
    }

    // Fetch old item for validation and values
    $stmt = $conn->prepare("SELECT id, sale_id, product_id, quantity, total_price, exchanged, refunded FROM sale_items WHERE id = ? AND sale_id = ? LIMIT 1");
    $stmt->bind_param('ii', $item_id, $sale_id);
    $stmt->execute();
    $oldItem = $stmt->get_result()->fetch_assoc();

    if (!$oldItem) {
        throw new Exception('Sale item not found');
    }
    if (!empty($oldItem['refunded'])) {
        throw new Exception('Item already refunded');
    }
    if (!empty($oldItem['exchanged'])) {
        throw new Exception('Item already exchanged');
    }

    // Fetch new product price
    $priceStmt = $conn->prepare("SELECT price FROM products WHERE id = ? LIMIT 1");
    $priceStmt->bind_param('i', $new_product_id);
    $priceStmt->execute();
    $productRow = $priceStmt->get_result()->fetch_assoc();
    if (!$productRow) {
        throw new Exception('New product not found');
    }
    $new_unit_price = (float)$productRow['price'];
    $new_total = $new_unit_price * $new_quantity;

    // Mark original item as exchanged
    $stmt = $conn->prepare("UPDATE sale_items SET exchanged = 1 WHERE id = ? AND sale_id = ?");
    $stmt->bind_param('ii', $item_id, $sale_id);
    if (!$stmt->execute()) {
        throw new Exception('Failed to mark item as exchanged');
    }

    // Restock old product quantity
    $restockStmt = $conn->prepare("UPDATE products SET stock = stock + ? WHERE id = ?");
    $restockStmt->bind_param('ii', $oldItem['quantity'], $oldItem['product_id']);
    if (!$restockStmt->execute()) {
        throw new Exception('Failed to restock old product');
    }

    // Add new sale item for exchanged product
    $stmt2 = $conn->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, total_price) VALUES (?, ?, ?, ?, ?)");
    $stmt2->bind_param('iiidd', $sale_id, $new_product_id, $new_quantity, $new_unit_price, $new_total);
    if (!$stmt2->execute()) {
        throw new Exception('Failed to add replacement item');
    }

    // Update inventory for new product
    $stmt3 = $conn->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
    $stmt3->bind_param('ii', $new_quantity, $new_product_id);
    if (!$stmt3->execute()) {
        throw new Exception('Failed to update inventory');
    }

    // Adjust sale total by price difference (keeps invoice consistent)
    $totalAdjStmt = $conn->prepare("UPDATE sales SET total_amount = total_amount + ? WHERE id = ?");
    $totalAdjStmt->bind_param('di', $price_diff, $sale_id);
    $totalAdjStmt->execute();

    // Commission adjustment: remove old item commission, add new item commission
    $commissionStmt = $conn->prepare("SELECT id, commission_amount FROM sales_commissions WHERE sale_id = ? ORDER BY id DESC LIMIT 1");
    $commission_delta = ($new_total - $oldItem['total_price']) * $commission_rate;
    if ($commissionStmt) {
        $commissionStmt->bind_param('i', $sale_id);
        $commissionStmt->execute();
        $commissionRow = $commissionStmt->get_result()->fetch_assoc();
        if ($commissionRow) {
            $current_commission = (float)$commissionRow['commission_amount'];
            $new_commission = $current_commission + $commission_delta;
            if ($new_commission < 0) {
                $new_commission = 0;
            }
            $updateCommission = $conn->prepare("UPDATE sales_commissions SET commission_amount = ? WHERE id = ?");
            if ($updateCommission) {
                $updateCommission->bind_param('di', $new_commission, $commissionRow['id']);
                if (!$updateCommission->execute()) {
                    throw new Exception('Failed to update commission');
                }
            }
            @file_put_contents(__DIR__ . '/commission_log.txt', date('Y-m-d H:i:s') . " - Exchange commission adjust: sale {$sale_id}, item {$item_id}, delta {$commission_delta}, new {$new_commission}\n", FILE_APPEND);
        }
    }

    // Log the exchange in exchange_log table
    $stmtLog = $conn->prepare("INSERT INTO exchange_log (sale_id, old_item_id, new_product_id, new_quantity, price_diff) VALUES (?, ?, ?, ?, ?)");
    if ($stmtLog) {
        $stmtLog->bind_param('iiiid', $sale_id, $item_id, $new_product_id, $new_quantity, $price_diff);
        $stmtLog->execute();
        $stmtLog->close();
    }

    $conn->commit();
    echo json_encode(['success' => true, 'price_diff' => $price_diff]);
    exit;
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    exit;
}
