<?php
date_default_timezone_set('Asia/Colombo');
include 'config.php';

// Create a monitoring table to track when products disappear
$create_monitor = "
CREATE TABLE IF NOT EXISTS product_monitor (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id INT,
    product_name VARCHAR(100),
    product_barcode VARCHAR(50),
    check_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    status ENUM('exists', 'missing') DEFAULT 'exists',
    notes TEXT
)";
$conn->query($create_monitor);

echo "=== PRODUCT DISAPPEARANCE MONITOR ===\n\n";

// Record all current products for tracking
echo "Setting up monitoring for current products...\n";

$current_products = $conn->query("SELECT id, name, barcode FROM products");
$monitored_count = 0;

// Clear previous monitoring data
$conn->query("DELETE FROM product_monitor");

while ($product = $current_products->fetch_assoc()) {
    $stmt = $conn->prepare("INSERT INTO product_monitor (product_id, product_name, product_barcode, status) VALUES (?, ?, ?, 'exists')");
    $stmt->bind_param("iss", $product['id'], $product['name'], $product['barcode']);
    $stmt->execute();
    $monitored_count++;
}

echo "✅ Now monitoring $monitored_count products\n\n";

// Show what we're monitoring
echo "Current products being monitored:\n";
$monitor_list = $conn->query("SELECT product_id, product_name, product_barcode FROM product_monitor ORDER BY product_id DESC LIMIT 10");
while ($row = $monitor_list->fetch_assoc()) {
    echo "  ID {$row['product_id']}: {$row['product_name']} | {$row['product_barcode']}\n";
}

echo "\n=== INSTRUCTIONS ===\n";
echo "1. Add a new product through your web interface now\n";
echo "2. Run this script again to add the new product to monitoring\n";
echo "3. Check tomorrow - this script will tell you exactly which products disappeared\n";
echo "4. This will help us catch the exact moment products vanish\n\n";

// Check if any previously monitored products are now missing
echo "=== CHECKING FOR MISSING PRODUCTS ===\n";
$check_missing = $conn->query("
    SELECT pm.product_id, pm.product_name, pm.product_barcode, pm.check_time
    FROM product_monitor pm
    LEFT JOIN products p ON pm.product_id = p.id
    WHERE p.id IS NULL AND pm.status = 'exists'
");

if ($check_missing->num_rows > 0) {
    echo "❌ MISSING PRODUCTS DETECTED:\n";
    while ($missing = $check_missing->fetch_assoc()) {
        echo "  Lost: ID {$missing['product_id']} | {$missing['product_name']} | {$missing['product_barcode']} | Last seen: {$missing['check_time']}\n";
        
        // Update status to missing
        $update_stmt = $conn->prepare("UPDATE product_monitor SET status = 'missing', notes = 'Detected missing on monitor check' WHERE product_id = ?");
        $update_stmt->bind_param("i", $missing['product_id']);
        $update_stmt->execute();
    }
} else {
    echo "✅ All monitored products still exist\n";
}

echo "\n=== MONITOR FILE CREATED ===\n";
echo "This monitoring system will now track when products disappear.\n";
echo "Run this script daily to detect the exact timing of disappearances.\n";
?>