<?php
// Debug script for products functionality
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h2>Product Management Debug</h2>";

try {
    echo "<h3>1. Config File Check</h3>";
    if (file_exists('../config.php')) {
        echo "✓ config.php exists<br>";
        include '../config.php';
        echo "✓ config.php included<br>";
    } else {
        echo "✗ config.php not found<br>";
        exit;
    }

    echo "<h3>2. Database Connection Check</h3>";
    if ($conn && !$conn->connect_error) {
        echo "✓ Database connected successfully<br>";
        echo "Connected to database: " . htmlspecialchars($db) . "<br>";
    } else {
        echo "✗ Database connection failed: " . ($conn->connect_error ?? 'Unknown error') . "<br>";
        exit;
    }

    echo "<h3>3. Auth Check</h3>";
    if (file_exists('../auth.php')) {
        echo "✓ auth.php exists<br>";
        include '../auth.php';
        echo "✓ auth.php included<br>";
        
        // Start session if not started
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        
        if (isset($_SESSION['user_id'])) {
            echo "✓ User authenticated: ID = " . $_SESSION['user_id'] . "<br>";
        } else {
            echo "⚠ User not authenticated (session may be expired)<br>";
        }
    } else {
        echo "✗ auth.php not found<br>";
    }

    echo "<h3>4. Products Table Structure</h3>";
    $result = $conn->query("DESCRIBE products");
    if ($result) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['Field']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td>" . htmlspecialchars($row['Default'] ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "✗ Could not describe products table: " . $conn->error . "<br>";
    }

    echo "<h3>5. Sample Product Count</h3>";
    $count_result = $conn->query("SELECT COUNT(*) as count FROM products");
    if ($count_result) {
        $count = $count_result->fetch_assoc()['count'];
        echo "✓ Products table has {$count} products<br>";
    } else {
        echo "✗ Could not count products: " . $conn->error . "<br>";
    }

    echo "<h3>6. Recent Products (last 5)</h3>";
    $recent_result = $conn->query("SELECT id, name, price, stock FROM products ORDER BY id DESC LIMIT 5");
    if ($recent_result && $recent_result->num_rows > 0) {
        echo "<table border='1' cellpadding='5'>";
        echo "<tr><th>ID</th><th>Name</th><th>Price</th><th>Stock</th></tr>";
        while ($row = $recent_result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . $row['id'] . "</td>";
            echo "<td>" . htmlspecialchars($row['name']) . "</td>";
            echo "<td>" . number_format($row['price'], 2) . "</td>";
            echo "<td>" . $row['stock'] . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "No products found or query failed: " . $conn->error . "<br>";
    }

    echo "<h3>7. Test Delete Simulation (Safe)</h3>";
    if (isset($_GET['test_delete']) && is_numeric($_GET['test_delete'])) {
        $test_id = (int)$_GET['test_delete'];
        
        // Just check if the product exists (don't actually delete)
        $check_stmt = $conn->prepare("SELECT id, name FROM products WHERE id = ?");
        if ($check_stmt) {
            $check_stmt->bind_param('i', $test_id);
            $check_stmt->execute();
            $result = $check_stmt->get_result();
            
            if ($result->num_rows > 0) {
                $product = $result->fetch_assoc();
                echo "✓ Product found for deletion test: " . htmlspecialchars($product['name']) . " (ID: {$product['id']})<br>";
                echo "✓ Delete preparation would succeed<br>";
            } else {
                echo "✗ Product ID {$test_id} not found<br>";
            }
            $check_stmt->close();
        } else {
            echo "✗ Could not prepare test statement: " . $conn->error . "<br>";
        }
    } else {
        echo "Add ?test_delete=ID to URL to test delete preparation for a specific product ID<br>";
    }

} catch (Exception $e) {
    echo "✗ Exception: " . $e->getMessage() . "<br>";
} catch (Error $e) {
    echo "✗ PHP Error: " . $e->getMessage() . "<br>";
}

echo "<br><a href='products.php'>← Back to Products</a>";
?>